# QHCA SymPy / QuTiP Simulation Examples
# Author: Hiroko Konishi & Noa (assistant)
# Python >=3.9, packages: numpy, sympy, qutip, matplotlib
#
# Experiments:
#  (A) Temporal Recursive Superposition (SymPy)
#  (B) Entanglement Gradient on a 1D Chain (QuTiP)
#  (C) Entropy-Tuned Decoherence & CHSH (QuTiP)
#
# Run:
#   python qhca_sympy_qutip_examples.py
#
# Notes:
# - QuTiP is optional for (A). Required for (B) and (C).
# - Figures will be saved under ./figs/

import os
import math
import numpy as np

# Ensure figs dir
os.makedirs("figs", exist_ok=True)

# -----------------------------
# (A) Temporal Recursive Superposition (SymPy)
# -----------------------------
from sympy import symbols, Matrix, sqrt, Abs, lambdify

def experiment_A(T=25, alpha=0.3, beta=0.6, gamma=0.1):
    """
    Temporal recursive superposition:
        |psi_t> = alpha |psi_{t-1}> + beta |psi_t> + gamma |psi_{t+1}>
    We implement a forward-time recurrence by rearranging and normalizing:
        |psi_{t+1}> ∝ (1/gamma)|psi_t> - (alpha/gamma)|psi_{t-1}> + (-beta/gamma)|psi_t>
                     = ((1 - beta)/gamma)|psi_t> - (alpha/gamma)|psi_{t-1}>
    Using a 2D toy "consciousness subspace" basis to visualize overlaps.
    """
    # Use a 2D Hilbert space basis for illustration
    e0 = Matrix([1, 0])
    e1 = Matrix([0, 1])
    # Initial states
    psi_minus = e0
    psi_0 = (e0 + e1)
    psi_0 = psi_0 / sqrt((psi_0.T * psi_0)[0])

    seq = [psi_minus, psi_0]
    for t in range(T):
        # forward recurrence
        num = ((1 - beta)/gamma) * seq[-1] - (alpha/gamma) * seq[-2]
        # normalize
        norm = sqrt((num.T * num)[0])
        psi_next = num / norm
        seq.append(psi_next)

    # Overlap metrics
    overlaps = []
    for t in range(1, len(seq)-1):
        o = Abs((seq[t].T * seq[t+1])[0])  # |<psi_t | psi_{t+1}>|
        overlaps.append(float(o))

    # Save a simple plot (matplotlib)
    import matplotlib.pyplot as plt
    plt.figure()
    plt.plot(range(len(overlaps)), overlaps, marker='o')
    plt.xlabel("t")
    plt.ylabel(r"|<psi_t | psi_{t+1}>|")
    plt.title("Temporal Self-Interference Overlap (Experiment A)")
    plt.tight_layout()
    plt.savefig("figs/experiment_A_overlap.png")
    plt.close()

    return overlaps

# -----------------------------
# (B) Entanglement Gradient on a 1D Chain (QuTiP)
# -----------------------------
def experiment_B(N=6, t=1.0, J0=1.0):
    """
    Build a 1D chain of N qubits. Apply local H gates and a ZZ coupling with a spatially varying profile
    J_i = J0 * (1 + i/(N-1)) to induce an "entanglement gradient". Compute single-site von Neumann entropies S_i
    and approximate gradient via finite differences.
    """
    try:
        from qutip import tensor, basis, sigmax, sigmaz, qeye, H as hadamard_transform, Qobj, entropy_vn, propagator
    except Exception as e:
        print("[Experiment B] QuTiP not available:", e)
        return None

    # Initial |0..0>
    psi0 = tensor(*[basis(2,0) for _ in range(N)])

    # Local Hadamards
    H1 = (1/math.sqrt(2)) * Qobj(np.array([[1, 1],[1,-1]], dtype=complex))
    U_H_all = tensor(*[H1 for _ in range(N)])
    psi = U_H_all * psi0

    # ZZ coupling Hamiltonian with spatial profile J_i
    def embed(op, site, N):
        ops = []
        for i in range(N):
            if i == site:
                ops.append(op)
            else:
                ops.append(qeye(2))
        return tensor(*ops)

    ZZ_terms = 0
    for i in range(N-1):
        Ji = J0 * (1 + i/(N-1))
        ZZ_terms += Ji * embed(sigmaz(), i, N) * embed(sigmaz(), i+1, N)
    H_int = ZZ_terms

    # Time evolution
    U = (-1j * H_int * t).expm()
    psi_t = U * psi

    # Single-site entropies
    entropies = []
    for i in range(N):
        # partial trace over all but site i
        keep = [i]
        dims = [2] * N
        rho_i = psi_t.ptrace(keep)
        S_i = entropy_vn(rho_i, base=2)
        entropies.append(S_i)

    # Compute finite-difference gradient
    grad = [0.0] * N
    for i in range(1, N-1):
        grad[i] = 0.5 * (entropies[i+1] - entropies[i-1])
    grad[0] = entropies[1] - entropies[0]
    grad[-1] = entropies[-1] - entropies[-2]

    # Plot entropies and gradient
    import matplotlib.pyplot as plt
    x = np.arange(N)

    plt.figure()
    plt.plot(x, entropies, marker='o')
    plt.xlabel("site i")
    plt.ylabel("S_i (bits)")
    plt.title("Single-Site Entropy (Experiment B)")
    plt.tight_layout()
    plt.savefig("figs/experiment_B_entropy.png")
    plt.close()

    plt.figure()
    plt.plot(x, grad, marker='s')
    plt.xlabel("site i")
    plt.ylabel("dS/di")
    plt.title("Entanglement Gradient (Experiment B)")
    plt.tight_layout()
    plt.savefig("figs/experiment_B_gradient.png")
    plt.close()

    return entropies, grad

# -----------------------------
# (C) Entropy-Tuned Decoherence & CHSH (QuTiP)
# -----------------------------
def experiment_C(num_points=21):
    """
    Start from |Phi+> Bell state. Apply dephasing channel with strength gamma \in [0, 1].
    Compute CHSH S parameter; observe threshold S/sqrt(2) > 0.707.
    We sweep gamma, plot S vs gamma.
    """
    try:
        from qutip import tensor, basis, sigmax, sigmay, sigmaz, qeye, Qobj, kraus_map, ket2dm, expect
    except Exception as e:
        print("[Experiment C] QuTiP not available:", e)
        return None

    # Bell state |Phi+> = (|00> + |11>)/sqrt(2)
    zero = basis(2,0)
    one  = basis(2,1)
    bell = (tensor(zero, zero) + tensor(one, one)).unit()
    rho0 = ket2dm(bell)

    def dephase_channel(rho, gamma):
        """
        Single-qubit phase damping channel with parameter gamma (0..1).
        Apply to both qubits independently.
        Kraus ops: K0 = [[1, 0],[0, sqrt(1-gamma)]], K1 = [[0, 0],[0, sqrt(gamma)]]
        """
        K0 = Qobj(np.array([[1, 0],[0, np.sqrt(1-gamma)]], dtype=complex))
        K1 = Qobj(np.array([[0, 0],[0, np.sqrt(gamma)]], dtype=complex))
        # Apply to each qubit (local dephasing)
        K = [tensor(K0, K0), tensor(K0, K1), tensor(K1, K0), tensor(K1, K1)]
        out = sum([K_i * rho * K_i.dag() for K_i in K])
        return out

    # CHSH operators (A, A', B, B') on Pauli axes
    def chsh_S(rho):
        # Standard settings for Bell states
        from qutip import sigmax, sigmaz, qeye, tensor, expect
        X = tensor(sigmax(), qeye(2))
        Z = tensor(sigmaz(), qeye(2))
        X2 = tensor(qeye(2), sigmax())
        Z2 = tensor(qeye(2), sigmaz())

        B  = (X2 + Z2) / math.sqrt(2)
        Bp = (X2 - Z2) / math.sqrt(2)

        EXB   = expect(X * B,  rho)
        EZB   = expect(Z * B,  rho)
        EXBp  = expect(X * Bp, rho)
        EZBp  = expect(Z * Bp, rho)

        S = EXB + EZB + EXBp - EZBp
        return float(np.real(S))

    gammas = np.linspace(0.0, 1.0, num_points)
    S_vals = []
    for g in gammas:
        rho = dephase_channel(rho0, g)
        S_vals.append(chsh_S(rho))

    # Plot S vs gamma
    import matplotlib.pyplot as plt
    plt.figure()
    plt.plot(gammas, S_vals, marker='o')
    plt.axhline(2.0, linestyle='--')  # classical bound
    plt.xlabel("dephasing gamma")
    plt.ylabel("CHSH S")
    plt.title("CHSH vs Dephasing (Experiment C)")
    plt.tight_layout()
    plt.savefig("figs/experiment_C_CHSH.png")
    plt.close()

    return gammas, S_vals

if __name__ == "__main__":
    print("Running Experiment A (SymPy)...")
    A_overlaps = experiment_A()
    print("A overlaps[0..4]:", A_overlaps[:5])

    print("Running Experiment B (QuTiP)...")
    B_out = experiment_B()
    if B_out is not None:
        ent, grad = B_out
        print("B entropies:", ent)
        print("B gradient:", grad)

    print("Running Experiment C (QuTiP)...")
    C_out = experiment_C()
    if C_out is not None:
        gammas, S_vals = C_out
        print("C CHSH S (first 5):", S_vals[:5])

    print("Done. Figures saved under ./figs")
